﻿namespace Hims.Api.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Threading.Tasks;
    using Domain.Services;
    using Hims.Api.Models;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Microsoft.CodeAnalysis.CSharp;
    using Npgsql;
    using Shared.DataFilters;
    using Shared.EntityModels;
    using Shared.Library.Enums;
    using Shared.UserModels.Filters;
    using Utilities;

    /// <inheritdoc />
    [Authorize]
    [Route("api/rooms")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class RoomController : BaseController
    {
        /// <summary>
        /// The coupon services.
        /// </summary>
        private readonly IRoomService roomService;

        /// <summary>
        /// The audit log services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <inheritdoc />
        public RoomController(IRoomService roomService, IAuditLogService auditLogServices)
        {
            this.roomService = roomService;
            this.auditLogServices = auditLogServices;
        }

        /// <summary>
        /// The fetch coupons.
        /// </summary>
        /// <param name="model">
        /// The coupon filter model.
        /// </param>
        /// <returns>
        /// The list of coupons.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of coupons.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("fetch")]
        [ProducesResponseType(typeof(List<RoomModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAsync([FromBody] RoomFilterModel model)
        {
            model = (RoomFilterModel)EmptyFilter.Handler(model);
            model.LocationId = model.LocationId > 0 ? model.LocationId : null;
            var wards = await this.roomService.FetchAsync(model);
            return wards == null ? this.ServerError() : this.Success(wards);
        }

        /// <summary>
        /// The fetch coupons.
        /// </summary>
        /// <param name="model">
        /// The coupon filter model.
        /// </param>
        /// <returns>
        /// The list of coupons.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of coupons.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("fetch-room")]
        [ProducesResponseType(typeof(List<RoomModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchRoomAsync([FromBody] RoomFilterModel model)
        {
            model = (RoomFilterModel)EmptyFilter.Handler(model); 
            var response = await this.roomService.FetchRoomAsync(model);
            return response == null ? this.ServerError() : this.Success(response);
        }


        /// <summary>
        /// The add coupon.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Coupon added successfully.
        /// - 409 - Coupon already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("add")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)] //
        public async Task<ActionResult> AddAsync([FromBody] RoomModel model, [FromHeader] LocationHeader header)
        {
            model = (RoomModel)EmptyFilter.Handler(model);
            var response = await this.roomService.AddAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given room has already been exists with us.");
                case -2:
                    return this.Conflict("Unable to find Modules Master Id, Please try later");
                case 0:
                    return this.ServerError();
            }

            var auditLogModel = new AuditLogModel
            {
                AccountId = model.CreatedBy,
                LogTypeId = (int)LogTypes.Room,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LogDescription = $" { model.ModifiedByName } has added Room - {model.RoomName} on { DateTime.UtcNow.AddMinutes(330) }",
                LocationId=Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success("Room has been added successfully.");
        }


        /// <summary>
        /// The update coupon.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Coupon updated successfully.
        /// - 409 - Coupon already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPut]
        [Route("update")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> UpdateAsync([FromBody] RoomModel model, [FromHeader] LocationHeader header)
        {
            model = (RoomModel)EmptyFilter.Handler(model);
            var response = await this.roomService.UpdateAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given Room has already been exists with us.");
                case 0:
                    return this.ServerError();
            }

            var auditLogModel = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                LogTypeId = (int)LogTypes.Room,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LogDescription = $" { model.ModifiedByName } has Updated Room - {model.RoomName} on { DateTime.UtcNow.AddMinutes(330) }",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success("Room has been updated successfully.");
        }

        /// <summary>
        /// The delete specialization.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Specialization deleted successfully.
        /// - 409 - Specialization can not be deleted.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("delete")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> DeleteAsync([FromBody] RoomModel model, [FromHeader] LocationHeader header)
        {
            try
            {
                model = (RoomModel)EmptyFilter.Handler(model);
                var roomId = await this.roomService.FindNameByRoomId(model.RoomId);
                var response = await this.roomService.DeleteAsync(model.RoomId);
                switch (response)
                {
                    case -1:
                        return this.Conflict("Unable to Find room");
                    case -2:
                        return this.Conflict("Room's Bed is allocated, please Deallocate then try");
                    case 0:
                        return this.ServerError();
                }

                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.ModifiedBy,
                    LogTypeId = (int)LogTypes.Room,
                    LogFrom = (int)AccountType.Administrator,
                    LogDate = DateTime.UtcNow.AddMinutes(330),
                    LogDescription = $" { model.ModifiedByName } has Deleted Room - {model.RoomName} on { DateTime.UtcNow.AddMinutes(330) }",
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLogModel);

                return this.Success("Room has been deleted successfully.");
            }
            catch (NpgsqlException exception)
            {
                if (exception.Message.Contains("violates foreign key constraint"))
                {
                    return this.Conflict("Room can't be deleted. Please contact Administrator.");
                }

                return this.ServerError();
            }
        }

        /// <summary>
        /// Modifies the ot room asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("modify-ot-room")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> ModifyOTRoomAsync([FromBody] RoomModel model, [FromHeader] LocationHeader header)
        {            
                model = (RoomModel)EmptyFilter.Handler(model);
                model.LocationId = int.Parse(header.LocationId);
                var response = model.OTRoomId == 0 ? await this.roomService.InsertOTRoom(model) : await this.roomService.UpdateOTRoom(model);
                if (response == 0)
                {
                    return this.ServerError();
                }

                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.OTRoom,
                    LogFrom = (int)AccountType.Administrator,
                    LogDate = DateTime.UtcNow.AddMinutes(330),
                    LogDescription = $@" { model.CreatedByName } has {(model.OTRoomId > 0 ? "Updated" : "Added")} OTRoom <b>{model.RoomName}</b> on { DateTime.UtcNow.AddMinutes(330) }",
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLogModel);;
                return this.Success(response);
        }

        /// <summary>
        /// Fetches the ot rooms asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("fetch-ot-room")]
        [ProducesResponseType(typeof(List<RoomModel>), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchOTRoomsAsync([FromBody] RoomFilterModel model, [FromHeader] LocationHeader header)
        {
            model ??= new RoomFilterModel();            
            model.LocationId = int.Parse(header.LocationId);
            var response = await this.roomService.FetchOTRoomAsync(model);
            return this.Success(response);
        }

        [HttpPost]
        [Route("modify-status")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> ModifyStatusAsync([FromBody] RoomModel model, [FromHeader] LocationHeader header)
        {
            model = (RoomModel)EmptyFilter.Handler(model);
            var response = await this.roomService.ActivateOrDeactivateTest(model);
            if (response == 0)
            {
                return this.ServerError();
            }
            if (response > 0)
            {
                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.ModifiedBy,
                    LogTypeId = (int)LogTypes.OTRoom,
                    LogFrom = (int)AccountType.Administrator,
                    LogDate = DateTime.UtcNow.AddMinutes(330),
                    LogDescription = $@"<b>{model.CreatedByName}</b> has {((bool)model.Active ? "Activated" : "Deactivated")} the <b>{model.RoomName}  </b> on { DateTime.UtcNow.AddMinutes(330) } successfully",
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLogModel);
            }

            return this.Success(response);

        }
        /// <summary>
        /// Fetches the ot rooms asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("fetch-surgery")]
        [ProducesResponseType(typeof(List<RoomModel>), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchSurgereisAsync([FromBody] RoomFilterModel model,[FromHeader] LocationHeader header)
        {
            model.LocationId = int.Parse(header.LocationId);
            var response = await this.roomService.FetchSurgeryOTRoom(model);
            return this.Success(response);
        }

        /// <summary>
        /// Changes the status asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="header">The header.</param>
        /// <returns></returns>
        [HttpPut]
        [Route("change-room-status")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> ChangeStatusAsync([FromBody] RoomModel model, [FromHeader] LocationHeader header)
        {
            model = (RoomModel)EmptyFilter.Handler(model);
            var response = await this.roomService.ChangeRoomStatusAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Unable to Find room");
                case -2:
                    return this.Conflict("Room's Bed is allocated, please Deallocate then try");
                case 0:
                    return this.ServerError();
            }

            var auditLogModel = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                LogTypeId = (int)LogTypes.Room,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LogDescription = $" {model.ModifiedByName} has changed status of  Room - {model.RoomName}  to {(model.Active ? "Active" : "Deactivate")  } on { DateTime.UtcNow.AddMinutes(330) }",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success("Room Status has been updated successfully.");
        }

    }
}